<?php
/**
 * Controller de Autenticação
 * Gerencia login, registro e logout
 */

require_once __DIR__ . '/../config/config.php';

class AuthController {
    private $userModel;
    
    public function __construct() {
        $this->userModel = new User();
    }
    
    /**
     * Exibe página de login
     */
    public function showLogin() {
        if (isLoggedIn()) {
            redirect(BASE_URL . '/dashboard');
        }
        
        require_once ROOT_PATH . '/views/auth/login.php';
    }
    
    /**
     * Processa login
     */
    public function login() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            redirect(BASE_URL . '/login');
        }
        
        $emailOrUsername = sanitize($_POST['email_username'] ?? '');
        $password = $_POST['password'] ?? '';
        
        // Validação
        $errors = [];
        
        if (empty($emailOrUsername)) {
            $errors[] = 'Email ou usuário é obrigatório';
        }
        
        if (empty($password)) {
            $errors[] = 'Senha é obrigatória';
        }
        
        if (!empty($errors)) {
            $_SESSION['errors'] = $errors;
            redirect(BASE_URL . '/login');
        }
        
        // Autenticação
        $user = $this->userModel->authenticate($emailOrUsername, $password);
        
        if (!$user) {
            setFlashMessage('Email/usuário ou senha incorretos', 'error');
            redirect(BASE_URL . '/login');
        }
        
        // Inicia sessão
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['username'] = $user['username'];
        $_SESSION['user_email'] = $user['email'];
        
        setFlashMessage('Bem-vindo, ' . $user['full_name'] . '!', 'success');
        redirect(BASE_URL . '/dashboard');
    }
    
    /**
     * Exibe página de registro
     */
    public function showRegister() {
        if (isLoggedIn()) {
            redirect(BASE_URL . '/dashboard');
        }
        
        require_once ROOT_PATH . '/views/auth/register.php';
    }
    
    /**
     * Processa registro
     */
    public function register() {
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            redirect(BASE_URL . '/register');
        }
        
        $data = [
            'username' => sanitize($_POST['username'] ?? ''),
            'email' => sanitize($_POST['email'] ?? ''),
            'full_name' => sanitize($_POST['full_name'] ?? ''),
            'password' => $_POST['password'] ?? '',
            'password_confirm' => $_POST['password_confirm'] ?? ''
        ];
        
        // Validação
        $errors = $this->validateRegistration($data);
        
        if (!empty($errors)) {
            $_SESSION['errors'] = $errors;
            $_SESSION['old_input'] = $data;
            redirect(BASE_URL . '/register');
        }
        
        // Cria usuário
        $userId = $this->userModel->create($data);
        
        if (!$userId) {
            setFlashMessage('Erro ao criar conta. Tente novamente.', 'error');
            redirect(BASE_URL . '/register');
        }
        
        // Faz login automático
        $_SESSION['user_id'] = $userId;
        $_SESSION['username'] = $data['username'];
        $_SESSION['user_email'] = $data['email'];
        
        setFlashMessage('Conta criada com sucesso! Bem-vindo ao HeroCat!', 'success');
        redirect(BASE_URL . '/dashboard/setup-pet');
    }
    
    /**
     * Valida dados de registro
     */
    private function validateRegistration($data) {
        $errors = [];
        
        // Username
        if (empty($data['username'])) {
            $errors[] = 'Usuário é obrigatório';
        } elseif (strlen($data['username']) < 3) {
            $errors[] = 'Usuário deve ter no mínimo 3 caracteres';
        } elseif (!preg_match('/^[a-zA-Z0-9_]+$/', $data['username'])) {
            $errors[] = 'Usuário deve conter apenas letras, números e underscore';
        } elseif ($this->userModel->usernameExists($data['username'])) {
            $errors[] = 'Este usuário já está em uso';
        }
        
        // Email
        if (empty($data['email'])) {
            $errors[] = 'Email é obrigatório';
        } elseif (!validateEmail($data['email'])) {
            $errors[] = 'Email inválido';
        } elseif ($this->userModel->emailExists($data['email'])) {
            $errors[] = 'Este email já está cadastrado';
        }
        
        // Nome completo
        if (empty($data['full_name'])) {
            $errors[] = 'Nome completo é obrigatório';
        } elseif (strlen($data['full_name']) < 3) {
            $errors[] = 'Nome deve ter no mínimo 3 caracteres';
        }
        
        // Senha
        if (empty($data['password'])) {
            $errors[] = 'Senha é obrigatória';
        } elseif (strlen($data['password']) < PASSWORD_MIN_LENGTH) {
            $errors[] = 'Senha deve ter no mínimo ' . PASSWORD_MIN_LENGTH . ' caracteres';
        }
        
        // Confirmação de senha
        if ($data['password'] !== $data['password_confirm']) {
            $errors[] = 'As senhas não coincidem';
        }
        
        return $errors;
    }
    
    /**
     * Logout
     */
    public function logout() {
        session_unset();
        session_destroy();
        
        setFlashMessage('Você saiu do sistema', 'info');
        redirect(BASE_URL . '/login');
    }
}

// Roteamento simples
if (isset($_GET['action'])) {
    $controller = new AuthController();
    $action = $_GET['action'];
    
    switch ($action) {
        case 'login':
            if ($_SERVER['REQUEST_METHOD'] === 'POST') {
                $controller->login();
            } else {
                $controller->showLogin();
            }
            break;
            
        case 'register':
            if ($_SERVER['REQUEST_METHOD'] === 'POST') {
                $controller->register();
            } else {
                $controller->showRegister();
            }
            break;
            
        case 'logout':
            $controller->logout();
            break;
            
        default:
            redirect(BASE_URL);
    }
}
