<?php
/**
 * Dashboard Controller
 * Gerencia área logada do usuário
 */

require_once __DIR__ . '/../config/config.php';

class DashboardController {
    private $userModel;
    private $petModel;
    private $walkModel;
    private $achievementModel;
    
    public function __construct() {
        // Verifica se está logado
        if (!isLoggedIn()) {
            redirect(BASE_URL . '/login');
        }
        
        $this->userModel = new User();
        $this->petModel = new Pet();
        $this->walkModel = new Walk();
        $this->achievementModel = new Achievement();
    }
    
    /**
     * Dashboard principal
     */
    public function index() {
        $userId = getCurrentUserId();
        
        // Dados do usuário
        $user = $this->userModel->findById($userId);
        $stats = $this->userModel->getStats($userId);
        $ranking = $this->userModel->getRanking($userId);
        
        // Pets do usuário
        $pets = $this->petModel->findByUserId($userId);
        
        // Caminhadas recentes
        $recentWalks = $this->walkModel->findByUserId($userId, 5);
        
        // Conquistas recentes
        $achievements = $this->achievementModel->getUserAchievements($userId);
        
        // Nível
        $currentLevel = getLevelFromXP($stats['total_experience'] ?? 0);
        $nextLevelXP = getXPForNextLevel($currentLevel);
        $levelProgress = getLevelProgress($stats['total_experience'] ?? 0, $currentLevel);
        
        require_once ROOT_PATH . '/views/dashboard/index.php';
    }
    
    /**
     * Página de pets
     */
    public function pets() {
        $userId = getCurrentUserId();
        $pets = $this->petModel->findByUserId($userId);
        
        require_once ROOT_PATH . '/views/dashboard/pets.php';
    }
    
    /**
     * Adicionar novo pet
     */
    public function addPet() {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $userId = getCurrentUserId();
            
            $data = [
                'user_id' => $userId,
                'name' => sanitize($_POST['name'] ?? ''),
                'type' => sanitize($_POST['type'] ?? ''),
                'breed' => sanitize($_POST['breed'] ?? ''),
                'age' => intval($_POST['age'] ?? 0),
                'weight' => floatval($_POST['weight'] ?? 0)
            ];
            
            // Validação
            $errors = [];
            
            if (empty($data['name'])) {
                $errors[] = 'Nome do pet é obrigatório';
            }
            
            if (!in_array($data['type'], ['dog', 'cat'])) {
                $errors[] = 'Tipo de pet inválido';
            }
            
            if (!empty($errors)) {
                $_SESSION['errors'] = $errors;
                redirect(BASE_URL . '/dashboard/pets');
            }
            
            // Upload de avatar se fornecido
            if (!empty($_FILES['avatar']['name'])) {
                $validationErrors = validateImageUpload($_FILES['avatar']);
                if (empty($validationErrors)) {
                    $filename = uploadImage($_FILES['avatar'], PET_AVATAR_PATH);
                    if ($filename) {
                        $data['avatar'] = $filename;
                    }
                }
            }
            
            $petId = $this->petModel->create($data);
            
            if ($petId) {
                setFlashMessage('Pet adicionado com sucesso!', 'success');
            } else {
                setFlashMessage('Erro ao adicionar pet', 'error');
            }
            
            redirect(BASE_URL . '/dashboard/pets');
        }
        
        require_once ROOT_PATH . '/views/dashboard/add-pet.php';
    }
    
    /**
     * Página de conquistas
     */
    public function achievements() {
        $userId = getCurrentUserId();
        
        // Conquistas desbloqueadas
        $unlockedAchievements = $this->achievementModel->getUserAchievements($userId);
        
        // Progresso de todas as conquistas
        $progress = $this->achievementModel->getProgress($userId);
        
        require_once ROOT_PATH . '/views/dashboard/achievements.php';
    }
    
    /**
     * Histórico de caminhadas
     */
    public function history() {
        $userId = getCurrentUserId();
        $page = intval($_GET['page'] ?? 1);
        $limit = ITEMS_PER_PAGE;
        $offset = ($page - 1) * $limit;
        
        $walks = $this->walkModel->findByUserId($userId, $limit, $offset);
        
        require_once ROOT_PATH . '/views/dashboard/history.php';
    }
    
    /**
     * Detalhes de uma caminhada
     */
    public function walkDetails() {
        $userId = getCurrentUserId();
        $walkId = intval($_GET['id'] ?? 0);
        
        if (!$walkId) {
            setFlashMessage('Caminhada não encontrada', 'error');
            redirect(BASE_URL . '/dashboard/history');
        }
        
        $walk = $this->walkModel->findById($walkId);
        
        if (!$walk || $walk['user_id'] != $userId) {
            setFlashMessage('Caminhada não encontrada', 'error');
            redirect(BASE_URL . '/dashboard/history');
        }
        
        $gpsLocations = $this->walkModel->getGPSLocations($walkId);
        
        require_once ROOT_PATH . '/views/dashboard/walk-details.php';
    }
    
    /**
     * Página de estatísticas
     */
    public function statistics() {
        $userId = getCurrentUserId();
        
        $stats = $this->userModel->getStats($userId);
        
        // Estatísticas mensais
        $year = intval($_GET['year'] ?? date('Y'));
        $month = intval($_GET['month'] ?? date('m'));
        $monthlyStats = $this->walkModel->getMonthlyStats($userId, $year, $month);
        
        require_once ROOT_PATH . '/views/dashboard/statistics.php';
    }
    
    /**
     * Ranking
     */
    public function ranking() {
        $topUsers = $this->userModel->getTopUsers(50);
        $userId = getCurrentUserId();
        $userRanking = $this->userModel->getRanking($userId);
        
        require_once ROOT_PATH . '/views/dashboard/ranking.php';
    }
    
    /**
     * Perfil do usuário
     */
    public function profile() {
        $userId = getCurrentUserId();
        $user = $this->userModel->findById($userId);
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $data = [
                'full_name' => sanitize($_POST['full_name'] ?? ''),
                'email' => sanitize($_POST['email'] ?? ''),
                'username' => sanitize($_POST['username'] ?? '')
            ];
            
            // Validação
            $errors = [];
            
            if (empty($data['full_name'])) {
                $errors[] = 'Nome completo é obrigatório';
            }
            
            if (empty($data['email']) || !validateEmail($data['email'])) {
                $errors[] = 'Email inválido';
            } elseif ($data['email'] != $user['email'] && $this->userModel->emailExists($data['email'])) {
                $errors[] = 'Este email já está em uso';
            }
            
            if (empty($data['username'])) {
                $errors[] = 'Usuário é obrigatório';
            } elseif ($data['username'] != $user['username'] && $this->userModel->usernameExists($data['username'])) {
                $errors[] = 'Este usuário já está em uso';
            }
            
            // Atualizar senha se fornecida
            if (!empty($_POST['new_password'])) {
                if (strlen($_POST['new_password']) < PASSWORD_MIN_LENGTH) {
                    $errors[] = 'Senha deve ter no mínimo ' . PASSWORD_MIN_LENGTH . ' caracteres';
                } elseif ($_POST['new_password'] !== $_POST['confirm_password']) {
                    $errors[] = 'As senhas não coincidem';
                } else {
                    $data['password'] = $_POST['new_password'];
                }
            }
            
            // Upload de avatar
            if (!empty($_FILES['avatar']['name'])) {
                $validationErrors = validateImageUpload($_FILES['avatar']);
                if (empty($validationErrors)) {
                    $filename = uploadImage($_FILES['avatar'], AVATAR_PATH);
                    if ($filename) {
                        $data['avatar'] = $filename;
                    }
                } else {
                    $errors = array_merge($errors, $validationErrors);
                }
            }
            
            if (!empty($errors)) {
                $_SESSION['errors'] = $errors;
                redirect(BASE_URL . '/dashboard/profile');
            }
            
            if ($this->userModel->update($userId, $data)) {
                setFlashMessage('Perfil atualizado com sucesso!', 'success');
                $_SESSION['username'] = $data['username'];
            } else {
                setFlashMessage('Erro ao atualizar perfil', 'error');
            }
            
            redirect(BASE_URL . '/dashboard/profile');
        }
        
        require_once ROOT_PATH . '/views/dashboard/profile.php';
    }
    
    /**
     * Página de caminhada (iniciar nova)
     */
    public function walk() {
        $userId = getCurrentUserId();
        $pets = $this->petModel->findByUserId($userId);
        
        if (empty($pets)) {
            setFlashMessage('Você precisa cadastrar um pet primeiro!', 'warning');
            redirect(BASE_URL . '/dashboard/add-pet');
        }
        
        require_once ROOT_PATH . '/views/dashboard/walk.php';
    }
}

// Roteamento
$controller = new DashboardController();
$action = $_GET['page'] ?? 'index';

switch ($action) {
    case 'index':
    case 'dashboard':
        $controller->index();
        break;
    
    case 'pets':
        $controller->pets();
        break;
    
    case 'add-pet':
    case 'setup-pet':
        $controller->addPet();
        break;
    
    case 'achievements':
        $controller->achievements();
        break;
    
    case 'history':
        $controller->history();
        break;
    
    case 'walk-details':
        $controller->walkDetails();
        break;
    
    case 'statistics':
    case 'stats':
        $controller->statistics();
        break;
    
    case 'ranking':
    case 'leaderboard':
        $controller->ranking();
        break;
    
    case 'profile':
        $controller->profile();
        break;
    
    case 'walk':
        $controller->walk();
        break;
    
    default:
        $controller->index();
}
