<?php
/**
 * API Controller para Caminhadas
 * Endpoints para iniciar, rastrear e finalizar caminhadas
 */

require_once __DIR__ . '/../config/config.php';

header('Content-Type: application/json');

// Verifica se está logado
if (!isLoggedIn()) {
    jsonResponse(['error' => 'Não autenticado'], 401);
}

$walkModel = new Walk();
$petModel = new Pet();
$userId = getCurrentUserId();

$action = $_GET['action'] ?? '';
$method = $_SERVER['REQUEST_METHOD'];

switch ($action) {
    /**
     * POST /api/walks?action=start
     * Inicia uma nova caminhada
     */
    case 'start':
        if ($method !== 'POST') {
            jsonResponse(['error' => 'Método não permitido'], 405);
        }
        
        $data = json_decode(file_get_contents('php://input'), true);
        $petId = $data['pet_id'] ?? null;
        
        // Validação
        if (!$petId) {
            jsonResponse(['error' => 'Pet ID é obrigatório'], 400);
        }
        
        // Verifica se pet pertence ao usuário
        if (!$petModel->belongsToUser($petId, $userId)) {
            jsonResponse(['error' => 'Pet não encontrado'], 404);
        }
        
        // Inicia caminhada
        $walkId = $walkModel->start($userId, $petId);
        
        if (!$walkId) {
            jsonResponse(['error' => 'Erro ao iniciar caminhada'], 500);
        }
        
        jsonResponse([
            'success' => true,
            'walk_id' => $walkId,
            'message' => 'Caminhada iniciada!'
        ]);
        break;
    
    /**
     * POST /api/walks?action=track
     * Adiciona ponto GPS à caminhada
     */
    case 'track':
        if ($method !== 'POST') {
            jsonResponse(['error' => 'Método não permitido'], 405);
        }
        
        $data = json_decode(file_get_contents('php://input'), true);
        $walkId = $data['walk_id'] ?? null;
        $latitude = $data['latitude'] ?? null;
        $longitude = $data['longitude'] ?? null;
        
        // Validação
        if (!$walkId || !$latitude || !$longitude) {
            jsonResponse(['error' => 'Dados incompletos'], 400);
        }
        
        if (!validateCoordinates($latitude, $longitude)) {
            jsonResponse(['error' => 'Coordenadas inválidas'], 400);
        }
        
        // Verifica se caminhada pertence ao usuário
        $walk = $walkModel->findById($walkId);
        if (!$walk || $walk['user_id'] != $userId) {
            jsonResponse(['error' => 'Caminhada não encontrada'], 404);
        }
        
        // Adiciona localização
        $result = $walkModel->addGPSLocation($walkId, [
            'latitude' => $latitude,
            'longitude' => $longitude,
            'altitude' => $data['altitude'] ?? null,
            'speed' => $data['speed'] ?? null,
            'accuracy' => $data['accuracy'] ?? null
        ]);
        
        if (!$result) {
            jsonResponse(['error' => 'Erro ao salvar localização'], 500);
        }
        
        jsonResponse([
            'success' => true,
            'message' => 'Localização registrada'
        ]);
        break;
    
    /**
     * POST /api/walks?action=finish
     * Finaliza caminhada
     */
    case 'finish':
        if ($method !== 'POST') {
            jsonResponse(['error' => 'Método não permitido'], 405);
        }
        
        $data = json_decode(file_get_contents('php://input'), true);
        $walkId = $data['walk_id'] ?? null;
        
        // Validação
        if (!$walkId) {
            jsonResponse(['error' => 'Walk ID é obrigatório'], 400);
        }
        
        // Verifica se caminhada pertence ao usuário
        $walk = $walkModel->findById($walkId);
        if (!$walk || $walk['user_id'] != $userId) {
            jsonResponse(['error' => 'Caminhada não encontrada'], 404);
        }
        
        // Valida dados da caminhada
        $distance = floatval($data['distance'] ?? 0);
        $duration = intval($data['duration'] ?? 0);
        
        if (!isValidWalk($distance, $duration)) {
            jsonResponse([
                'error' => 'Caminhada inválida',
                'message' => 'Distância mínima: ' . MIN_WALK_DISTANCE . 'km, Duração mínima: ' . MIN_WALK_DURATION . ' minutos'
            ], 400);
        }
        
        // Calcula calorias se não fornecido
        $calories = $data['calories'] ?? calculateCalories($distance);
        
        // Finaliza caminhada
        $result = $walkModel->finish($walkId, [
            'distance' => $distance,
            'duration' => $duration,
            'calories' => $calories,
            'steps' => $data['steps'] ?? null,
            'route_data' => isset($data['route']) ? json_encode($data['route']) : null,
            'finished_at' => date('Y-m-d H:i:s')
        ]);
        
        if (!$result) {
            jsonResponse(['error' => 'Erro ao finalizar caminhada'], 500);
        }
        
        // Busca caminhada atualizada
        $walk = $walkModel->findById($walkId);
        
        // Busca pet atualizado
        $pet = $petModel->findById($walk['pet_id']);
        
        jsonResponse([
            'success' => true,
            'message' => 'Caminhada finalizada!',
            'walk' => $walk,
            'pet' => $pet,
            'xp_gained' => $walk['experience_gained']
        ]);
        break;
    
    /**
     * GET /api/walks?action=history
     * Histórico de caminhadas
     */
    case 'history':
        if ($method !== 'GET') {
            jsonResponse(['error' => 'Método não permitido'], 405);
        }
        
        $page = intval($_GET['page'] ?? 1);
        $limit = intval($_GET['limit'] ?? ITEMS_PER_PAGE);
        $offset = ($page - 1) * $limit;
        
        $walks = $walkModel->findByUserId($userId, $limit, $offset);
        
        jsonResponse([
            'success' => true,
            'walks' => $walks,
            'page' => $page,
            'limit' => $limit
        ]);
        break;
    
    /**
     * GET /api/walks?action=details&id=123
     * Detalhes de uma caminhada
     */
    case 'details':
        if ($method !== 'GET') {
            jsonResponse(['error' => 'Método não permitido'], 405);
        }
        
        $walkId = intval($_GET['id'] ?? 0);
        
        if (!$walkId) {
            jsonResponse(['error' => 'Walk ID é obrigatório'], 400);
        }
        
        $walk = $walkModel->findById($walkId);
        
        if (!$walk || $walk['user_id'] != $userId) {
            jsonResponse(['error' => 'Caminhada não encontrada'], 404);
        }
        
        // Busca localizações GPS
        $gpsLocations = $walkModel->getGPSLocations($walkId);
        
        jsonResponse([
            'success' => true,
            'walk' => $walk,
            'gps_locations' => $gpsLocations
        ]);
        break;
    
    /**
     * GET /api/walks?action=stats
     * Estatísticas do usuário
     */
    case 'stats':
        if ($method !== 'GET') {
            jsonResponse(['error' => 'Método não permitido'], 405);
        }
        
        $userModel = new User();
        $stats = $userModel->getStats($userId);
        
        if (!$stats) {
            jsonResponse(['error' => 'Estatísticas não encontradas'], 404);
        }
        
        // Adiciona informações de nível
        $currentLevel = getLevelFromXP($stats['total_experience']);
        $nextLevelXP = getXPForNextLevel($currentLevel);
        $progress = getLevelProgress($stats['total_experience'], $currentLevel);
        
        jsonResponse([
            'success' => true,
            'stats' => $stats,
            'level' => [
                'current' => $currentLevel,
                'next_level_xp' => $nextLevelXP,
                'progress_percentage' => $progress
            ]
        ]);
        break;
    
    /**
     * GET /api/walks?action=monthly&year=2024&month=12
     * Estatísticas mensais
     */
    case 'monthly':
        if ($method !== 'GET') {
            jsonResponse(['error' => 'Método não permitido'], 405);
        }
        
        $year = intval($_GET['year'] ?? date('Y'));
        $month = intval($_GET['month'] ?? date('m'));
        
        $stats = $walkModel->getMonthlyStats($userId, $year, $month);
        
        jsonResponse([
            'success' => true,
            'year' => $year,
            'month' => $month,
            'stats' => $stats
        ]);
        break;
    
    default:
        jsonResponse(['error' => 'Ação inválida'], 400);
}
