-- HeroCat Database Schema
-- MySQL 5.7+

CREATE DATABASE IF NOT EXISTS herocat_db CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE herocat_db;

-- Tabela de Usuários
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL UNIQUE,
    email VARCHAR(100) NOT NULL UNIQUE,
    password VARCHAR(255) NOT NULL,
    full_name VARCHAR(100) NOT NULL,
    avatar VARCHAR(255) DEFAULT 'default-avatar.png',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    last_login TIMESTAMP NULL,
    is_active TINYINT(1) DEFAULT 1,
    INDEX idx_email (email),
    INDEX idx_username (username)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tabela de Pets
CREATE TABLE IF NOT EXISTS pets (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    name VARCHAR(50) NOT NULL,
    type ENUM('dog', 'cat') NOT NULL,
    breed VARCHAR(50),
    age INT,
    weight DECIMAL(5,2),
    avatar VARCHAR(255) DEFAULT 'default-pet.png',
    level INT DEFAULT 1,
    experience INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    is_active TINYINT(1) DEFAULT 1,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_level (level)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tabela de Caminhadas
CREATE TABLE IF NOT EXISTS walks (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    pet_id INT NOT NULL,
    distance DECIMAL(10,2) NOT NULL COMMENT 'Distância em quilômetros',
    duration INT NOT NULL COMMENT 'Duração em minutos',
    calories DECIMAL(8,2) COMMENT 'Calorias queimadas',
    steps INT COMMENT 'Número de passos',
    route_data TEXT COMMENT 'JSON com coordenadas GPS',
    started_at TIMESTAMP NOT NULL,
    finished_at TIMESTAMP NOT NULL,
    experience_gained INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (pet_id) REFERENCES pets(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_pet_id (pet_id),
    INDEX idx_started_at (started_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tabela de Localizações GPS (para rastreamento em tempo real)
CREATE TABLE IF NOT EXISTS gps_locations (
    id INT AUTO_INCREMENT PRIMARY KEY,
    walk_id INT NOT NULL,
    latitude DECIMAL(10,8) NOT NULL,
    longitude DECIMAL(11,8) NOT NULL,
    altitude DECIMAL(8,2),
    speed DECIMAL(5,2),
    accuracy DECIMAL(5,2),
    recorded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (walk_id) REFERENCES walks(id) ON DELETE CASCADE,
    INDEX idx_walk_id (walk_id),
    INDEX idx_recorded_at (recorded_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tabela de Conquistas
CREATE TABLE IF NOT EXISTS achievements (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    icon VARCHAR(255),
    category ENUM('distance', 'streak', 'level', 'special') NOT NULL,
    requirement_value INT NOT NULL COMMENT 'Valor necessário para conquistar',
    experience_reward INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_category (category)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tabela de Conquistas dos Usuários
CREATE TABLE IF NOT EXISTS user_achievements (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    achievement_id INT NOT NULL,
    unlocked_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (achievement_id) REFERENCES achievements(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_achievement (user_id, achievement_id),
    INDEX idx_user_id (user_id),
    INDEX idx_achievement_id (achievement_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tabela de Sequências (Streaks)
CREATE TABLE IF NOT EXISTS streaks (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    current_streak INT DEFAULT 0,
    longest_streak INT DEFAULT 0,
    last_walk_date DATE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_streak (user_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tabela de Estatísticas Diárias
CREATE TABLE IF NOT EXISTS daily_stats (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    date DATE NOT NULL,
    total_distance DECIMAL(10,2) DEFAULT 0,
    total_duration INT DEFAULT 0,
    total_steps INT DEFAULT 0,
    total_calories DECIMAL(8,2) DEFAULT 0,
    walks_count INT DEFAULT 0,
    experience_gained INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_date (user_id, date),
    INDEX idx_user_id (user_id),
    INDEX idx_date (date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Tabela de Níveis e XP
CREATE TABLE IF NOT EXISTS level_requirements (
    level INT PRIMARY KEY,
    experience_required INT NOT NULL,
    title VARCHAR(50),
    rewards JSON COMMENT 'Recompensas do nível em JSON'
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Inserir dados iniciais de conquistas
INSERT INTO achievements (name, description, icon, category, requirement_value, experience_reward) VALUES
('Primeiro Passo', 'Complete sua primeira caminhada', '🎯', 'special', 1, 50),
('Caminhante Iniciante', 'Caminhe 5 km no total', '🚶', 'distance', 5, 100),
('Maratonista', 'Caminhe 50 km no total', '🏃', 'distance', 50, 500),
('Ultra Maratonista', 'Caminhe 100 km no total', '⚡', 'distance', 100, 1000),
('Sequência de 7 dias', 'Mantenha uma sequência de 7 dias', '🔥', 'streak', 7, 200),
('Sequência de 30 dias', 'Mantenha uma sequência de 30 dias', '💪', 'streak', 30, 1000),
('Nível 5', 'Alcance o nível 5', '⭐', 'level', 5, 150),
('Nível 10', 'Alcance o nível 10', '🌟', 'level', 10, 300),
('Nível 20', 'Alcance o nível 20', '💫', 'level', 20, 750),
('Caminhada Matinal', 'Complete uma caminhada antes das 8h', '🌅', 'special', 1, 75);

-- Inserir dados iniciais de níveis
INSERT INTO level_requirements (level, experience_required, title, rewards) VALUES
(1, 0, 'Novato', '{"coins": 0}'),
(2, 100, 'Iniciante', '{"coins": 10}'),
(3, 250, 'Caminhante', '{"coins": 20}'),
(4, 500, 'Explorador', '{"coins": 30}'),
(5, 850, 'Aventureiro', '{"coins": 50, "badge": "explorer"}'),
(6, 1300, 'Viajante', '{"coins": 70}'),
(7, 1850, 'Desbravador', '{"coins": 90}'),
(8, 2500, 'Mestre Caminhante', '{"coins": 120}'),
(9, 3250, 'Guardião', '{"coins": 150}'),
(10, 4100, 'Herói', '{"coins": 200, "badge": "hero"}'),
(11, 5050, 'Lenda Iniciante', '{"coins": 250}'),
(12, 6100, 'Lenda', '{"coins": 300}'),
(13, 7250, 'Super Lenda', '{"coins": 350}'),
(14, 8500, 'Ultra Lenda', '{"coins": 400}'),
(15, 10000, 'Mito', '{"coins": 500, "badge": "myth"}'),
(16, 12000, 'Imortal', '{"coins": 600}'),
(17, 14500, 'Divino', '{"coins": 700}'),
(18, 17500, 'Celestial', '{"coins": 800}'),
(19, 21000, 'Supremo', '{"coins": 900}'),
(20, 25000, 'Lendário', '{"coins": 1000, "badge": "legendary"}');

-- Views úteis para relatórios

-- View de estatísticas totais por usuário
CREATE OR REPLACE VIEW user_total_stats AS
SELECT 
    u.id as user_id,
    u.username,
    COUNT(DISTINCT w.id) as total_walks,
    COALESCE(SUM(w.distance), 0) as total_distance,
    COALESCE(SUM(w.duration), 0) as total_duration,
    COALESCE(SUM(w.steps), 0) as total_steps,
    COALESCE(SUM(w.calories), 0) as total_calories,
    COALESCE(SUM(w.experience_gained), 0) as total_experience,
    COUNT(DISTINCT ua.achievement_id) as achievements_unlocked,
    COALESCE(s.current_streak, 0) as current_streak,
    COALESCE(s.longest_streak, 0) as longest_streak
FROM users u
LEFT JOIN walks w ON u.id = w.user_id
LEFT JOIN user_achievements ua ON u.id = ua.user_id
LEFT JOIN streaks s ON u.id = s.user_id
GROUP BY u.id, u.username, s.current_streak, s.longest_streak;

-- View de ranking de usuários
CREATE OR REPLACE VIEW user_ranking AS
SELECT 
    u.id,
    u.username,
    u.full_name,
    u.avatar,
    COALESCE(SUM(w.distance), 0) as total_distance,
    COALESCE(SUM(w.experience_gained), 0) as total_experience,
    COUNT(DISTINCT ua.achievement_id) as achievements_count,
    RANK() OVER (ORDER BY COALESCE(SUM(w.experience_gained), 0) DESC) as ranking_position
FROM users u
LEFT JOIN walks w ON u.id = w.user_id
LEFT JOIN user_achievements ua ON u.id = ua.user_id
WHERE u.is_active = 1
GROUP BY u.id, u.username, u.full_name, u.avatar
ORDER BY total_experience DESC;
