<?php
/**
 * Funções Auxiliares
 * HeroCat System
 */

/**
 * Sanitiza string para prevenir XSS
 */
function sanitize($data) {
    return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
}

/**
 * Valida email
 */
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

/**
 * Hash de senha
 */
function hashPassword($password) {
    return password_hash($password, PASSWORD_BCRYPT, ['cost' => 12]);
}

/**
 * Verifica senha
 */
function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * Gera token aleatório
 */
function generateToken($length = 32) {
    return bin2hex(random_bytes($length));
}

/**
 * Redireciona para uma URL
 */
function redirect($url) {
    header("Location: " . $url);
    exit();
}

/**
 * Verifica se usuário está logado
 */
function isLoggedIn() {
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

/**
 * Retorna ID do usuário logado
 */
function getCurrentUserId() {
    return $_SESSION['user_id'] ?? null;
}

/**
 * Define mensagem flash
 */
function setFlashMessage($message, $type = 'info') {
    $_SESSION['flash_message'] = $message;
    $_SESSION['flash_type'] = $type;
}

/**
 * Obtém e limpa mensagem flash
 */
function getFlashMessage() {
    if (isset($_SESSION['flash_message'])) {
        $message = $_SESSION['flash_message'];
        $type = $_SESSION['flash_type'] ?? 'info';
        unset($_SESSION['flash_message'], $_SESSION['flash_type']);
        return ['message' => $message, 'type' => $type];
    }
    return null;
}

/**
 * Formata distância
 */
function formatDistance($km) {
    if ($km < 1) {
        return round($km * 1000) . ' m';
    }
    return round($km, 2) . ' km';
}

/**
 * Formata duração em minutos para horas e minutos
 */
function formatDuration($minutes) {
    if ($minutes < 60) {
        return $minutes . ' min';
    }
    $hours = floor($minutes / 60);
    $mins = $minutes % 60;
    return $hours . 'h ' . $mins . 'min';
}

/**
 * Formata data em português
 */
function formatDate($date, $format = 'd/m/Y H:i') {
    if (!$date) return '-';
    $timestamp = is_numeric($date) ? $date : strtotime($date);
    return date($format, $timestamp);
}

/**
 * Formata data relativa (há X tempo)
 */
function timeAgo($datetime) {
    $timestamp = is_numeric($datetime) ? $datetime : strtotime($datetime);
    $diff = time() - $timestamp;
    
    if ($diff < 60) return 'agora mesmo';
    if ($diff < 3600) return floor($diff / 60) . ' min atrás';
    if ($diff < 86400) return floor($diff / 3600) . ' h atrás';
    if ($diff < 604800) return floor($diff / 86400) . ' dias atrás';
    if ($diff < 2592000) return floor($diff / 604800) . ' semanas atrás';
    if ($diff < 31536000) return floor($diff / 2592000) . ' meses atrás';
    return floor($diff / 31536000) . ' anos atrás';
}

/**
 * Calcula XP baseado em distância e duração
 */
function calculateXP($distance, $duration) {
    $xp = ($distance * XP_PER_KM) + ($duration * XP_PER_MINUTE) + DAILY_WALK_BONUS;
    return (int) round($xp);
}

/**
 * Calcula calorias queimadas (estimativa)
 */
function calculateCalories($distance, $weight = 70) {
    // Fórmula aproximada: 0.57 * peso * distância
    return round(0.57 * $weight * $distance, 2);
}

/**
 * Valida upload de imagem
 */
function validateImageUpload($file) {
    $errors = [];
    
    if ($file['error'] !== UPLOAD_ERR_OK) {
        $errors[] = 'Erro no upload do arquivo';
        return $errors;
    }
    
    if ($file['size'] > MAX_UPLOAD_SIZE) {
        $errors[] = 'Arquivo muito grande. Máximo: ' . (MAX_UPLOAD_SIZE / 1024 / 1024) . 'MB';
    }
    
    if (!in_array($file['type'], ALLOWED_IMAGE_TYPES)) {
        $errors[] = 'Tipo de arquivo não permitido';
    }
    
    return $errors;
}

/**
 * Faz upload de imagem
 */
function uploadImage($file, $destination) {
    $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
    $filename = uniqid() . '_' . time() . '.' . $ext;
    $filepath = $destination . '/' . $filename;
    
    if (!is_dir($destination)) {
        mkdir($destination, 0755, true);
    }
    
    if (move_uploaded_file($file['tmp_name'], $filepath)) {
        return $filename;
    }
    
    return false;
}

/**
 * Retorna nível baseado em XP
 */
function getLevelFromXP($xp) {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->prepare("SELECT level FROM level_requirements WHERE experience_required <= ? ORDER BY level DESC LIMIT 1");
    $stmt->execute([$xp]);
    $result = $stmt->fetch();
    return $result ? $result['level'] : 1;
}

/**
 * Retorna XP necessário para próximo nível
 */
function getXPForNextLevel($currentLevel) {
    $db = Database::getInstance()->getConnection();
    $stmt = $db->prepare("SELECT experience_required FROM level_requirements WHERE level = ?");
    $stmt->execute([$currentLevel + 1]);
    $result = $stmt->fetch();
    return $result ? $result['experience_required'] : 0;
}

/**
 * Retorna progresso percentual para próximo nível
 */
function getLevelProgress($currentXP, $currentLevel) {
    $db = Database::getInstance()->getConnection();
    
    // XP necessário para o nível atual
    $stmt = $db->prepare("SELECT experience_required FROM level_requirements WHERE level = ?");
    $stmt->execute([$currentLevel]);
    $currentLevelXP = $stmt->fetch()['experience_required'] ?? 0;
    
    // XP necessário para o próximo nível
    $stmt->execute([$currentLevel + 1]);
    $nextLevelXP = $stmt->fetch()['experience_required'] ?? $currentXP;
    
    $xpInLevel = $currentXP - $currentLevelXP;
    $xpNeeded = $nextLevelXP - $currentLevelXP;
    
    if ($xpNeeded <= 0) return 100;
    
    return min(100, round(($xpInLevel / $xpNeeded) * 100));
}

/**
 * Resposta JSON
 */
function jsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit();
}

/**
 * Log de erros
 */
function logError($message, $context = []) {
    $logFile = ROOT_PATH . '/logs/error.log';
    $logDir = dirname($logFile);
    
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    $timestamp = date('Y-m-d H:i:s');
    $contextStr = !empty($context) ? json_encode($context) : '';
    $logMessage = "[$timestamp] $message $contextStr\n";
    
    error_log($logMessage, 3, $logFile);
}

/**
 * Valida coordenadas GPS
 */
function validateCoordinates($lat, $lng) {
    return is_numeric($lat) && is_numeric($lng) 
        && $lat >= -90 && $lat <= 90 
        && $lng >= -180 && $lng <= 180;
}

/**
 * Calcula distância entre dois pontos GPS (em km)
 */
function calculateDistance($lat1, $lon1, $lat2, $lon2) {
    $earthRadius = 6371; // Raio da Terra em km
    
    $dLat = deg2rad($lat2 - $lat1);
    $dLon = deg2rad($lon2 - $lon1);
    
    $a = sin($dLat/2) * sin($dLat/2) +
         cos(deg2rad($lat1)) * cos(deg2rad($lat2)) *
         sin($dLon/2) * sin($dLon/2);
    
    $c = 2 * atan2(sqrt($a), sqrt(1-$a));
    $distance = $earthRadius * $c;
    
    return $distance;
}

/**
 * Verifica se caminhada é válida
 */
function isValidWalk($distance, $duration) {
    return $distance >= MIN_WALK_DISTANCE && $duration >= MIN_WALK_DURATION;
}

/**
 * Proteção CSRF - Gera token
 */
function generateCSRFToken() {
    if (empty($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * Proteção CSRF - Verifica token
 */
function verifyCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Formata número com separador de milhares
 */
function formatNumber($number, $decimals = 0) {
    return number_format($number, $decimals, ',', '.');
}
