<?php
/**
 * Model Achievement
 * Gerencia conquistas e badges
 */

class Achievement {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    /**
     * Busca todas as conquistas
     */
    public function getAll() {
        $stmt = $this->db->query("SELECT * FROM achievements ORDER BY category, requirement_value");
        return $stmt->fetchAll();
    }
    
    /**
     * Busca conquista por ID
     */
    public function findById($id) {
        $stmt = $this->db->prepare("SELECT * FROM achievements WHERE id = ?");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }
    
    /**
     * Busca conquistas do usuário
     */
    public function getUserAchievements($userId) {
        $sql = "SELECT a.*, ua.unlocked_at 
                FROM user_achievements ua
                INNER JOIN achievements a ON ua.achievement_id = a.id
                WHERE ua.user_id = ?
                ORDER BY ua.unlocked_at DESC";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$userId]);
        return $stmt->fetchAll();
    }
    
    /**
     * Verifica se usuário tem conquista
     */
    public function hasAchievement($userId, $achievementId) {
        $stmt = $this->db->prepare("SELECT COUNT(*) FROM user_achievements WHERE user_id = ? AND achievement_id = ?");
        $stmt->execute([$userId, $achievementId]);
        return $stmt->fetchColumn() > 0;
    }
    
    /**
     * Desbloqueia conquista para usuário
     */
    public function unlock($userId, $achievementId) {
        try {
            // Verifica se já tem
            if ($this->hasAchievement($userId, $achievementId)) {
                return false;
            }
            
            $this->db->beginTransaction();
            
            // Insere conquista
            $sql = "INSERT INTO user_achievements (user_id, achievement_id) VALUES (?, ?)";
            $stmt = $this->db->prepare($sql);
            $stmt->execute([$userId, $achievementId]);
            
            // Busca recompensa de XP
            $achievement = $this->findById($achievementId);
            if ($achievement && $achievement['experience_reward'] > 0) {
                // Adiciona XP ao primeiro pet do usuário (ou poderia ser ao usuário)
                $petModel = new Pet();
                $pets = $petModel->findByUserId($userId);
                if (!empty($pets)) {
                    $petModel->updateLevelAndXP($pets[0]['id'], $achievement['experience_reward']);
                }
            }
            
            $this->db->commit();
            return true;
            
        } catch (PDOException $e) {
            $this->db->rollBack();
            logError("Erro ao desbloquear conquista: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Obtém progresso das conquistas
     */
    public function getProgress($userId) {
        $userModel = new User();
        $stats = $userModel->getStats($userId);
        
        if (!$stats) {
            return [];
        }
        
        $achievements = $this->getAll();
        $progress = [];
        
        foreach ($achievements as $achievement) {
            $current = 0;
            $unlocked = $this->hasAchievement($userId, $achievement['id']);
            
            switch ($achievement['category']) {
                case 'distance':
                    $current = $stats['total_distance'];
                    break;
                    
                case 'streak':
                    $current = $stats['current_streak'];
                    break;
                    
                case 'level':
                    $current = getLevelFromXP($stats['total_experience']);
                    break;
                    
                case 'special':
                    if ($achievement['name'] == 'Primeiro Passo') {
                        $current = $stats['total_walks'];
                    }
                    break;
            }
            
            $percentage = min(100, round(($current / $achievement['requirement_value']) * 100));
            
            $progress[] = [
                'achievement' => $achievement,
                'current' => $current,
                'required' => $achievement['requirement_value'],
                'percentage' => $percentage,
                'unlocked' => $unlocked
            ];
        }
        
        return $progress;
    }
    
    /**
     * Obtém conquistas recentes de todos os usuários
     */
    public function getRecentUnlocks($limit = 10) {
        $sql = "SELECT ua.*, u.username, u.avatar, a.name, a.icon 
                FROM user_achievements ua
                INNER JOIN users u ON ua.user_id = u.id
                INNER JOIN achievements a ON ua.achievement_id = a.id
                WHERE u.is_active = 1
                ORDER BY ua.unlocked_at DESC
                LIMIT ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$limit]);
        return $stmt->fetchAll();
    }
}
