<?php
/**
 * Model Pet
 * Gerencia pets dos usuários
 */

class Pet {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    /**
     * Cria novo pet
     */
    public function create($data) {
        try {
            $sql = "INSERT INTO pets (user_id, name, type, breed, age, weight, avatar) 
                    VALUES (:user_id, :name, :type, :breed, :age, :weight, :avatar)";
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute([
                ':user_id' => $data['user_id'],
                ':name' => $data['name'],
                ':type' => $data['type'],
                ':breed' => $data['breed'] ?? null,
                ':age' => $data['age'] ?? null,
                ':weight' => $data['weight'] ?? null,
                ':avatar' => $data['avatar'] ?? 'default-pet.png'
            ]);
            
            return $this->db->lastInsertId();
        } catch (PDOException $e) {
            logError("Erro ao criar pet: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Busca pet por ID
     */
    public function findById($id) {
        $stmt = $this->db->prepare("SELECT * FROM pets WHERE id = ? AND is_active = 1");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }
    
    /**
     * Busca pets do usuário
     */
    public function findByUserId($userId) {
        $stmt = $this->db->prepare("SELECT * FROM pets WHERE user_id = ? AND is_active = 1 ORDER BY created_at DESC");
        $stmt->execute([$userId]);
        return $stmt->fetchAll();
    }
    
    /**
     * Atualiza pet
     */
    public function update($id, $data) {
        try {
            $fields = [];
            $params = [];
            
            if (isset($data['name'])) {
                $fields[] = "name = ?";
                $params[] = $data['name'];
            }
            
            if (isset($data['type'])) {
                $fields[] = "type = ?";
                $params[] = $data['type'];
            }
            
            if (isset($data['breed'])) {
                $fields[] = "breed = ?";
                $params[] = $data['breed'];
            }
            
            if (isset($data['age'])) {
                $fields[] = "age = ?";
                $params[] = $data['age'];
            }
            
            if (isset($data['weight'])) {
                $fields[] = "weight = ?";
                $params[] = $data['weight'];
            }
            
            if (isset($data['avatar'])) {
                $fields[] = "avatar = ?";
                $params[] = $data['avatar'];
            }
            
            if (empty($fields)) return false;
            
            $params[] = $id;
            $sql = "UPDATE pets SET " . implode(', ', $fields) . " WHERE id = ?";
            
            $stmt = $this->db->prepare($sql);
            return $stmt->execute($params);
        } catch (PDOException $e) {
            logError("Erro ao atualizar pet: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Atualiza nível e experiência do pet
     */
    public function updateLevelAndXP($petId, $experience) {
        try {
            // Busca XP atual
            $stmt = $this->db->prepare("SELECT experience FROM pets WHERE id = ?");
            $stmt->execute([$petId]);
            $currentXP = $stmt->fetchColumn();
            
            $newXP = $currentXP + $experience;
            $newLevel = getLevelFromXP($newXP);
            
            $stmt = $this->db->prepare("UPDATE pets SET experience = ?, level = ? WHERE id = ?");
            return $stmt->execute([$newXP, $newLevel, $petId]);
        } catch (PDOException $e) {
            logError("Erro ao atualizar nível do pet: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Deleta pet (soft delete)
     */
    public function delete($id) {
        $stmt = $this->db->prepare("UPDATE pets SET is_active = 0 WHERE id = ?");
        return $stmt->execute([$id]);
    }
    
    /**
     * Verifica se pet pertence ao usuário
     */
    public function belongsToUser($petId, $userId) {
        $stmt = $this->db->prepare("SELECT COUNT(*) FROM pets WHERE id = ? AND user_id = ? AND is_active = 1");
        $stmt->execute([$petId, $userId]);
        return $stmt->fetchColumn() > 0;
    }
    
    /**
     * Obtém estatísticas do pet
     */
    public function getStats($petId) {
        $sql = "SELECT 
                    p.*,
                    COUNT(DISTINCT w.id) as total_walks,
                    COALESCE(SUM(w.distance), 0) as total_distance,
                    COALESCE(SUM(w.duration), 0) as total_duration,
                    COALESCE(SUM(w.experience_gained), 0) as total_xp_from_walks
                FROM pets p
                LEFT JOIN walks w ON p.id = w.pet_id
                WHERE p.id = ? AND p.is_active = 1
                GROUP BY p.id";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$petId]);
        return $stmt->fetch();
    }
    
    /**
     * Obtém pets mais ativos
     */
    public function getTopPets($limit = 10) {
        $sql = "SELECT 
                    p.*,
                    u.username,
                    COUNT(DISTINCT w.id) as total_walks,
                    COALESCE(SUM(w.distance), 0) as total_distance
                FROM pets p
                INNER JOIN users u ON p.user_id = u.id
                LEFT JOIN walks w ON p.id = w.pet_id
                WHERE p.is_active = 1 AND u.is_active = 1
                GROUP BY p.id
                ORDER BY total_distance DESC
                LIMIT ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$limit]);
        return $stmt->fetchAll();
    }
}
