<?php
/**
 * Model User
 * Gerencia usuários do sistema
 */

class User {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    /**
     * Cria novo usuário
     */
    public function create($data) {
        try {
            $sql = "INSERT INTO users (username, email, password, full_name) 
                    VALUES (:username, :email, :password, :full_name)";
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute([
                ':username' => $data['username'],
                ':email' => $data['email'],
                ':password' => hashPassword($data['password']),
                ':full_name' => $data['full_name']
            ]);
            
            return $this->db->lastInsertId();
        } catch (PDOException $e) {
            logError("Erro ao criar usuário: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Busca usuário por ID
     */
    public function findById($id) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE id = ? AND is_active = 1");
        $stmt->execute([$id]);
        return $stmt->fetch();
    }
    
    /**
     * Busca usuário por email
     */
    public function findByEmail($email) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE email = ? AND is_active = 1");
        $stmt->execute([$email]);
        return $stmt->fetch();
    }
    
    /**
     * Busca usuário por username
     */
    public function findByUsername($username) {
        $stmt = $this->db->prepare("SELECT * FROM users WHERE username = ? AND is_active = 1");
        $stmt->execute([$username]);
        return $stmt->fetch();
    }
    
    /**
     * Verifica se email já existe
     */
    public function emailExists($email, $excludeId = null) {
        $sql = "SELECT COUNT(*) FROM users WHERE email = ?";
        $params = [$email];
        
        if ($excludeId) {
            $sql .= " AND id != ?";
            $params[] = $excludeId;
        }
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchColumn() > 0;
    }
    
    /**
     * Verifica se username já existe
     */
    public function usernameExists($username, $excludeId = null) {
        $sql = "SELECT COUNT(*) FROM users WHERE username = ?";
        $params = [$username];
        
        if ($excludeId) {
            $sql .= " AND id != ?";
            $params[] = $excludeId;
        }
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute($params);
        return $stmt->fetchColumn() > 0;
    }
    
    /**
     * Atualiza perfil do usuário
     */
    public function update($id, $data) {
        try {
            $fields = [];
            $params = [];
            
            if (isset($data['full_name'])) {
                $fields[] = "full_name = ?";
                $params[] = $data['full_name'];
            }
            
            if (isset($data['email'])) {
                $fields[] = "email = ?";
                $params[] = $data['email'];
            }
            
            if (isset($data['username'])) {
                $fields[] = "username = ?";
                $params[] = $data['username'];
            }
            
            if (isset($data['avatar'])) {
                $fields[] = "avatar = ?";
                $params[] = $data['avatar'];
            }
            
            if (isset($data['password'])) {
                $fields[] = "password = ?";
                $params[] = hashPassword($data['password']);
            }
            
            if (empty($fields)) return false;
            
            $params[] = $id;
            $sql = "UPDATE users SET " . implode(', ', $fields) . " WHERE id = ?";
            
            $stmt = $this->db->prepare($sql);
            return $stmt->execute($params);
        } catch (PDOException $e) {
            logError("Erro ao atualizar usuário: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Atualiza último login
     */
    public function updateLastLogin($id) {
        $stmt = $this->db->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
        return $stmt->execute([$id]);
    }
    
    /**
     * Autenticação de usuário
     */
    public function authenticate($emailOrUsername, $password) {
        // Busca por email ou username
        $user = $this->findByEmail($emailOrUsername);
        if (!$user) {
            $user = $this->findByUsername($emailOrUsername);
        }
        
        if (!$user) {
            return false;
        }
        
        if (!verifyPassword($password, $user['password'])) {
            return false;
        }
        
        // Atualiza último login
        $this->updateLastLogin($user['id']);
        
        return $user;
    }
    
    /**
     * Obtém estatísticas do usuário
     */
    public function getStats($userId) {
        $stmt = $this->db->prepare("SELECT * FROM user_total_stats WHERE user_id = ?");
        $stmt->execute([$userId]);
        return $stmt->fetch();
    }
    
    /**
     * Obtém ranking do usuário
     */
    public function getRanking($userId) {
        $stmt = $this->db->prepare("SELECT ranking_position FROM user_ranking WHERE id = ?");
        $stmt->execute([$userId]);
        $result = $stmt->fetch();
        return $result ? $result['ranking_position'] : null;
    }
    
    /**
     * Obtém top usuários
     */
    public function getTopUsers($limit = 10) {
        $stmt = $this->db->prepare("SELECT * FROM user_ranking LIMIT ?");
        $stmt->execute([$limit]);
        return $stmt->fetchAll();
    }
    
    /**
     * Deleta usuário (soft delete)
     */
    public function delete($id) {
        $stmt = $this->db->prepare("UPDATE users SET is_active = 0 WHERE id = ?");
        return $stmt->execute([$id]);
    }
    
    /**
     * Obtém nível atual do usuário
     */
    public function getCurrentLevel($userId) {
        $stats = $this->getStats($userId);
        if (!$stats) return 1;
        
        return getLevelFromXP($stats['total_experience']);
    }
    
    /**
     * Obtém progresso do nível
     */
    public function getLevelProgress($userId) {
        $stats = $this->getStats($userId);
        if (!$stats) return 0;
        
        $currentLevel = getLevelFromXP($stats['total_experience']);
        return getLevelProgress($stats['total_experience'], $currentLevel);
    }
}
