<?php
/**
 * Model Walk
 * Gerencia caminhadas
 */

class Walk {
    private $db;
    
    public function __construct() {
        $this->db = Database::getInstance()->getConnection();
    }
    
    /**
     * Inicia uma nova caminhada
     */
    public function start($userId, $petId) {
        try {
            $sql = "INSERT INTO walks (user_id, pet_id, distance, duration, started_at, finished_at) 
                    VALUES (:user_id, :pet_id, 0, 0, NOW(), NOW())";
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute([
                ':user_id' => $userId,
                ':pet_id' => $petId
            ]);
            
            return $this->db->lastInsertId();
        } catch (PDOException $e) {
            logError("Erro ao iniciar caminhada: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Finaliza uma caminhada
     */
    public function finish($walkId, $data) {
        try {
            $this->db->beginTransaction();
            
            // Calcula XP
            $xp = calculateXP($data['distance'], $data['duration']);
            
            // Atualiza caminhada
            $sql = "UPDATE walks SET 
                    distance = :distance,
                    duration = :duration,
                    calories = :calories,
                    steps = :steps,
                    route_data = :route_data,
                    finished_at = :finished_at,
                    experience_gained = :experience
                    WHERE id = :id";
            
            $stmt = $this->db->prepare($sql);
            $stmt->execute([
                ':distance' => $data['distance'],
                ':duration' => $data['duration'],
                ':calories' => $data['calories'],
                ':steps' => $data['steps'] ?? null,
                ':route_data' => $data['route_data'] ?? null,
                ':finished_at' => $data['finished_at'] ?? date('Y-m-d H:i:s'),
                ':experience' => $xp,
                ':id' => $walkId
            ]);
            
            // Busca informações da caminhada
            $walk = $this->findById($walkId);
            
            // Atualiza XP do pet
            $petModel = new Pet();
            $petModel->updateLevelAndXP($walk['pet_id'], $xp);
            
            // Atualiza estatísticas diárias
            $this->updateDailyStats($walk['user_id'], $data);
            
            // Atualiza sequência
            $this->updateStreak($walk['user_id']);
            
            // Verifica conquistas
            $this->checkAchievements($walk['user_id']);
            
            $this->db->commit();
            return true;
            
        } catch (PDOException $e) {
            $this->db->rollBack();
            logError("Erro ao finalizar caminhada: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Adiciona localização GPS
     */
    public function addGPSLocation($walkId, $data) {
        try {
            $sql = "INSERT INTO gps_locations (walk_id, latitude, longitude, altitude, speed, accuracy) 
                    VALUES (:walk_id, :latitude, :longitude, :altitude, :speed, :accuracy)";
            
            $stmt = $this->db->prepare($sql);
            return $stmt->execute([
                ':walk_id' => $walkId,
                ':latitude' => $data['latitude'],
                ':longitude' => $data['longitude'],
                ':altitude' => $data['altitude'] ?? null,
                ':speed' => $data['speed'] ?? null,
                ':accuracy' => $data['accuracy'] ?? null
            ]);
        } catch (PDOException $e) {
            logError("Erro ao adicionar localização GPS: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Busca caminhada por ID
     */
    public function findById($id) {
        $sql = "SELECT w.*, p.name as pet_name, p.type as pet_type, u.username 
                FROM walks w
                INNER JOIN pets p ON w.pet_id = p.id
                INNER JOIN users u ON w.user_id = u.id
                WHERE w.id = ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$id]);
        return $stmt->fetch();
    }
    
    /**
     * Busca caminhadas do usuário
     */
    public function findByUserId($userId, $limit = 20, $offset = 0) {
        $sql = "SELECT w.*, p.name as pet_name, p.type as pet_type 
                FROM walks w
                INNER JOIN pets p ON w.pet_id = p.id
                WHERE w.user_id = ?
                ORDER BY w.started_at DESC
                LIMIT ? OFFSET ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$userId, $limit, $offset]);
        return $stmt->fetchAll();
    }
    
    /**
     * Busca caminhadas recentes
     */
    public function getRecentWalks($limit = 10) {
        $sql = "SELECT w.*, p.name as pet_name, p.type as pet_type, u.username, u.avatar 
                FROM walks w
                INNER JOIN pets p ON w.pet_id = p.id
                INNER JOIN users u ON w.user_id = u.id
                WHERE w.distance >= ?
                ORDER BY w.finished_at DESC
                LIMIT ?";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([MIN_WALK_DISTANCE, $limit]);
        return $stmt->fetchAll();
    }
    
    /**
     * Obtém localizações GPS de uma caminhada
     */
    public function getGPSLocations($walkId) {
        $stmt = $this->db->prepare("SELECT * FROM gps_locations WHERE walk_id = ? ORDER BY recorded_at ASC");
        $stmt->execute([$walkId]);
        return $stmt->fetchAll();
    }
    
    /**
     * Atualiza estatísticas diárias
     */
    private function updateDailyStats($userId, $data) {
        $date = date('Y-m-d');
        
        $sql = "INSERT INTO daily_stats (user_id, date, total_distance, total_duration, total_steps, total_calories, walks_count, experience_gained)
                VALUES (:user_id, :date, :distance, :duration, :steps, :calories, 1, :xp)
                ON DUPLICATE KEY UPDATE
                    total_distance = total_distance + :distance,
                    total_duration = total_duration + :duration,
                    total_steps = total_steps + :steps,
                    total_calories = total_calories + :calories,
                    walks_count = walks_count + 1,
                    experience_gained = experience_gained + :xp";
        
        $stmt = $this->db->prepare($sql);
        $xp = calculateXP($data['distance'], $data['duration']);
        
        return $stmt->execute([
            ':user_id' => $userId,
            ':date' => $date,
            ':distance' => $data['distance'],
            ':duration' => $data['duration'],
            ':steps' => $data['steps'] ?? 0,
            ':calories' => $data['calories'] ?? 0,
            ':xp' => $xp
        ]);
    }
    
    /**
     * Atualiza sequência (streak)
     */
    private function updateStreak($userId) {
        // Busca última caminhada
        $stmt = $this->db->prepare("SELECT MAX(DATE(started_at)) as last_date FROM walks WHERE user_id = ?");
        $stmt->execute([$userId]);
        $lastWalk = $stmt->fetch();
        
        if (!$lastWalk || !$lastWalk['last_date']) {
            return;
        }
        
        $today = date('Y-m-d');
        $yesterday = date('Y-m-d', strtotime('-1 day'));
        
        // Busca ou cria registro de streak
        $stmt = $this->db->prepare("SELECT * FROM streaks WHERE user_id = ?");
        $stmt->execute([$userId]);
        $streak = $stmt->fetch();
        
        if (!$streak) {
            // Cria novo streak
            $sql = "INSERT INTO streaks (user_id, current_streak, longest_streak, last_walk_date) 
                    VALUES (?, 1, 1, ?)";
            $stmt = $this->db->prepare($sql);
            $stmt->execute([$userId, $today]);
        } else {
            $currentStreak = $streak['current_streak'];
            $longestStreak = $streak['longest_streak'];
            
            // Se a última caminhada foi ontem, incrementa streak
            if ($streak['last_walk_date'] == $yesterday) {
                $currentStreak++;
                if ($currentStreak > $longestStreak) {
                    $longestStreak = $currentStreak;
                }
            }
            // Se não caminhou ontem, reseta streak
            elseif ($streak['last_walk_date'] != $today) {
                $currentStreak = 1;
            }
            
            $sql = "UPDATE streaks SET current_streak = ?, longest_streak = ?, last_walk_date = ? WHERE user_id = ?";
            $stmt = $this->db->prepare($sql);
            $stmt->execute([$currentStreak, $longestStreak, $today, $userId]);
        }
    }
    
    /**
     * Verifica e desbloqueia conquistas
     */
    private function checkAchievements($userId) {
        $achievementModel = new Achievement();
        $stats = (new User())->getStats($userId);
        
        if (!$stats) return;
        
        // Busca conquistas ainda não desbloqueadas
        $sql = "SELECT a.* FROM achievements a
                WHERE a.id NOT IN (
                    SELECT achievement_id FROM user_achievements WHERE user_id = ?
                )";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$userId]);
        $achievements = $stmt->fetchAll();
        
        foreach ($achievements as $achievement) {
            $unlocked = false;
            
            switch ($achievement['category']) {
                case 'distance':
                    if ($stats['total_distance'] >= $achievement['requirement_value']) {
                        $unlocked = true;
                    }
                    break;
                    
                case 'streak':
                    if ($stats['current_streak'] >= $achievement['requirement_value']) {
                        $unlocked = true;
                    }
                    break;
                    
                case 'level':
                    $level = getLevelFromXP($stats['total_experience']);
                    if ($level >= $achievement['requirement_value']) {
                        $unlocked = true;
                    }
                    break;
                    
                case 'special':
                    // Lógica especial para conquistas específicas
                    if ($achievement['name'] == 'Primeiro Passo' && $stats['total_walks'] >= 1) {
                        $unlocked = true;
                    }
                    break;
            }
            
            if ($unlocked) {
                $achievementModel->unlock($userId, $achievement['id']);
            }
        }
    }
    
    /**
     * Obtém estatísticas mensais
     */
    public function getMonthlyStats($userId, $year, $month) {
        $sql = "SELECT 
                    DATE(started_at) as date,
                    COUNT(*) as walks_count,
                    SUM(distance) as total_distance,
                    SUM(duration) as total_duration,
                    SUM(calories) as total_calories,
                    SUM(experience_gained) as total_xp
                FROM walks
                WHERE user_id = ? 
                AND YEAR(started_at) = ? 
                AND MONTH(started_at) = ?
                GROUP BY DATE(started_at)
                ORDER BY date ASC";
        
        $stmt = $this->db->prepare($sql);
        $stmt->execute([$userId, $year, $month]);
        return $stmt->fetchAll();
    }
    
    /**
     * Deleta caminhada
     */
    public function delete($id) {
        try {
            $stmt = $this->db->prepare("DELETE FROM walks WHERE id = ?");
            return $stmt->execute([$id]);
        } catch (PDOException $e) {
            logError("Erro ao deletar caminhada: " . $e->getMessage());
            return false;
        }
    }
}
